#!/usr/bin/env node

/**
 * reengage.email - Email Re-engagement Automation
 *
 * Automatically sends a 3-email sequence to inactive users.
 * Run with: node automation.js
 * Test with: node automation.js --dry-run
 */

require('dotenv').config();
const fs = require('fs');
const path = require('path');
const nodemailer = require('nodemailer');
const cron = require('node-cron');

// ============================================
// Configuration
// ============================================

const CONFIG_PATH = process.env.CONFIG_PATH || './config.json';
const DRY_RUN = process.argv.includes('--dry-run') || process.env.DRY_RUN === 'true';
const RUN_ONCE = process.argv.includes('--once');

let config;

function loadConfig() {
  try {
    const configFile = fs.readFileSync(CONFIG_PATH, 'utf8');
    config = JSON.parse(configFile);

    // Override with environment variables if present
    if (process.env.SMTP_HOST) {
      config.smtp = {
        host: process.env.SMTP_HOST,
        port: parseInt(process.env.SMTP_PORT) || 587,
        secure: process.env.SMTP_SECURE === 'true',
        auth: {
          user: process.env.SMTP_USER,
          pass: process.env.SMTP_PASS
        }
      };
    }

    if (process.env.TRIGGER_DAYS) {
      config.trigger.daysInactive = parseInt(process.env.TRIGGER_DAYS);
    }

    log('info', 'Configuration loaded successfully');
    return true;
  } catch (error) {
    log('error', `Failed to load config: ${error.message}`);
    log('error', 'Make sure config.json exists. Copy from config.example.json');
    return false;
  }
}

// ============================================
// Logging
// ============================================

function log(level, message, data = null) {
  const timestamp = new Date().toISOString();
  const prefix = `[${timestamp}] [${level.toUpperCase()}]`;

  if (data) {
    console.log(`${prefix} ${message}`, data);
  } else {
    console.log(`${prefix} ${message}`);
  }
}

// ============================================
// Data Management
// ============================================

function loadUsers() {
  const usersPath = config.data.usersFile;

  if (!fs.existsSync(usersPath)) {
    log('warn', `Users file not found: ${usersPath}`);
    log('info', 'Creating sample users file...');
    createSampleUsersFile(usersPath);
  }

  try {
    const content = fs.readFileSync(usersPath, 'utf8');
    const lines = content.trim().split('\n');
    const headers = lines[0].split(',').map(h => h.trim());

    const users = lines.slice(1).map(line => {
      const values = parseCSVLine(line);
      const user = {};
      headers.forEach((header, index) => {
        user[header] = values[index]?.trim() || '';
      });
      return user;
    }).filter(user => user.email); // Filter out empty rows

    log('info', `Loaded ${users.length} users`);
    return users;
  } catch (error) {
    log('error', `Failed to load users: ${error.message}`);
    return [];
  }
}

function parseCSVLine(line) {
  const result = [];
  let current = '';
  let inQuotes = false;

  for (let i = 0; i < line.length; i++) {
    const char = line[i];

    if (char === '"') {
      inQuotes = !inQuotes;
    } else if (char === ',' && !inQuotes) {
      result.push(current);
      current = '';
    } else {
      current += char;
    }
  }
  result.push(current);

  return result;
}

function createSampleUsersFile(filePath) {
  const sampleContent = `email,firstName,lastName,lastActivityDate
john@example.com,John,Doe,2024-01-01
jane@example.com,Jane,Smith,2024-01-15
`;

  const dir = path.dirname(filePath);
  if (!fs.existsSync(dir)) {
    fs.mkdirSync(dir, { recursive: true });
  }

  fs.writeFileSync(filePath, sampleContent);
  log('info', `Created sample users file: ${filePath}`);
}

function loadState() {
  const statePath = config.data.stateFile;

  if (!fs.existsSync(statePath)) {
    return { emailsSent: {} };
  }

  try {
    const content = fs.readFileSync(statePath, 'utf8');
    return JSON.parse(content);
  } catch (error) {
    log('warn', `Failed to load state, starting fresh: ${error.message}`);
    return { emailsSent: {} };
  }
}

function saveState(state) {
  const statePath = config.data.stateFile;

  const dir = path.dirname(statePath);
  if (!fs.existsSync(dir)) {
    fs.mkdirSync(dir, { recursive: true });
  }

  fs.writeFileSync(statePath, JSON.stringify(state, null, 2));
}

// ============================================
// Email Functions
// ============================================

function createTransporter() {
  return nodemailer.createTransport({
    host: config.smtp.host,
    port: config.smtp.port,
    secure: config.smtp.secure,
    auth: config.smtp.auth
  });
}

function loadTemplate(templatePath) {
  try {
    return fs.readFileSync(templatePath, 'utf8');
  } catch (error) {
    log('error', `Failed to load template: ${templatePath}`);
    return null;
  }
}

function renderTemplate(template, variables) {
  let rendered = template;

  for (const [key, value] of Object.entries(variables)) {
    const regex = new RegExp(`{{${key}}}`, 'g');
    rendered = rendered.replace(regex, value || '');
  }

  return rendered;
}

async function sendEmail(transporter, user, emailConfig, variables) {
  const template = loadTemplate(emailConfig.template);

  if (!template) {
    log('error', `Skipping email - template not found: ${emailConfig.template}`);
    return false;
  }

  const html = renderTemplate(template, variables);
  const subject = renderTemplate(emailConfig.subject, variables);

  const mailOptions = {
    from: `"${config.emails.fromName}" <${config.emails.fromEmail}>`,
    replyTo: config.emails.replyTo,
    to: user.email,
    subject: subject,
    html: html
  };

  if (DRY_RUN) {
    log('info', `[DRY RUN] Would send email #${emailConfig.id} to ${user.email}`);
    log('info', `  Subject: ${subject}`);
    return true;
  }

  try {
    await transporter.sendMail(mailOptions);
    log('info', `Sent email #${emailConfig.id} to ${user.email}`);
    return true;
  } catch (error) {
    log('error', `Failed to send email to ${user.email}: ${error.message}`);
    return false;
  }
}

// ============================================
// Core Logic
// ============================================

function isUserInactive(user) {
  const lastActivity = new Date(user.lastActivityDate);
  const now = new Date();
  const daysSinceActivity = Math.floor((now - lastActivity) / (1000 * 60 * 60 * 24));

  return daysSinceActivity >= config.trigger.daysInactive;
}

function getNextEmailForUser(user, state) {
  const userState = state.emailsSent[user.email] || { sentEmails: [], lastSentDate: null };
  const sentEmailIds = userState.sentEmails || [];

  // Find the next email in sequence that hasn't been sent
  for (const emailConfig of config.emails.sequence) {
    if (!sentEmailIds.includes(emailConfig.id)) {
      // Check if enough days have passed since last email
      if (userState.lastSentDate) {
        const lastSent = new Date(userState.lastSentDate);
        const now = new Date();
        const daysSinceLastEmail = Math.floor((now - lastSent) / (1000 * 60 * 60 * 24));

        if (daysSinceLastEmail < emailConfig.delayDays) {
          return null; // Not enough time has passed
        }
      }

      return emailConfig;
    }
  }

  return null; // All emails sent
}

function markEmailSent(state, userEmail, emailId) {
  if (!state.emailsSent[userEmail]) {
    state.emailsSent[userEmail] = { sentEmails: [], lastSentDate: null };
  }

  state.emailsSent[userEmail].sentEmails.push(emailId);
  state.emailsSent[userEmail].lastSentDate = new Date().toISOString();
}

async function runAutomation() {
  log('info', '========================================');
  log('info', 'Starting re-engagement automation run');
  log('info', `Trigger: ${config.trigger.type} (${config.trigger.daysInactive} days)`);

  if (DRY_RUN) {
    log('info', '>>> DRY RUN MODE - No emails will be sent <<<');
  }

  const users = loadUsers();
  const state = loadState();

  if (users.length === 0) {
    log('warn', 'No users found. Add users to your CSV file.');
    return;
  }

  const transporter = DRY_RUN ? null : createTransporter();

  let emailsSentCount = 0;
  let usersProcessed = 0;

  for (const user of users) {
    // Check if user is inactive
    if (!isUserInactive(user)) {
      continue;
    }

    usersProcessed++;

    // Get next email to send
    const nextEmail = getNextEmailForUser(user, state);

    if (!nextEmail) {
      continue; // No email to send (all sent or waiting for delay)
    }

    // Prepare template variables
    const variables = {
      firstName: user.firstName || 'there',
      lastName: user.lastName || '',
      email: user.email,
      businessName: config.business.name,
      websiteUrl: config.business.website,
      unsubscribeUrl: `${config.business.website}/unsubscribe?email=${encodeURIComponent(user.email)}`
    };

    // Send email
    const success = await sendEmail(transporter, user, nextEmail, variables);

    if (success) {
      markEmailSent(state, user.email, nextEmail.id);
      emailsSentCount++;
    }
  }

  // Save state
  if (!DRY_RUN) {
    saveState(state);
  }

  log('info', '----------------------------------------');
  log('info', `Run complete. Processed ${usersProcessed} inactive users.`);
  log('info', `Emails sent: ${emailsSentCount}`);
  log('info', '========================================');
}

// ============================================
// Main Entry Point
// ============================================

async function main() {
  console.log(`
  ╔═══════════════════════════════════════════╗
  ║         reengage.email automation         ║
  ║     Re-engage inactive users with email   ║
  ╚═══════════════════════════════════════════╝
  `);

  // Load configuration
  if (!loadConfig()) {
    process.exit(1);
  }

  // Run once mode
  if (RUN_ONCE || DRY_RUN) {
    await runAutomation();
    return;
  }

  // Scheduled mode
  if (config.schedule.enabled) {
    log('info', `Scheduling automation with cron: ${config.schedule.cron}`);

    cron.schedule(config.schedule.cron, async () => {
      await runAutomation();
    }, {
      timezone: config.options.timezone
    });

    log('info', 'Automation scheduler started. Press Ctrl+C to stop.');

    // Run immediately on start
    await runAutomation();
  } else {
    log('info', 'Scheduled mode disabled. Running once...');
    await runAutomation();
  }
}

// Handle errors gracefully
process.on('uncaughtException', (error) => {
  log('error', `Uncaught exception: ${error.message}`);
  process.exit(1);
});

process.on('unhandledRejection', (reason, promise) => {
  log('error', `Unhandled rejection: ${reason}`);
});

// Start
main().catch(error => {
  log('error', `Fatal error: ${error.message}`);
  process.exit(1);
});
